from flask import Flask, request, session, render_template, redirect, url_for, flash
import os
import auth
import db
from datetime import datetime

app = Flask(__name__)
app.secret_key = os.environ.get('SECRET_KEY', 'your-secret-key-here')

@app.route('/')
def index():
    return redirect(url_for('login'))

@app.route('/login', methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        username = request.form.get('username')
        password = request.form.get('password')
        user = auth.login(username, password)
        if user:
            session['user_id'] = user['id']
            session['role'] = user['role']
            if user['role'] == 'admin':
                return redirect(url_for('admin_dashboard'))
            else:
                return redirect(url_for('student_dashboard'))
        else:
            flash('Invalid credentials')
    return render_template('login.html')

@app.route('/logout')
def logout():
    session.clear()
    return redirect(url_for('login'))

@app.route('/admin/dashboard')
def admin_dashboard():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    return render_template('admin_dashboard.html')

@app.route('/student/dashboard')
def student_dashboard():
    if session.get('role') != 'student':
        return redirect(url_for('login'))
    return render_template('student_dashboard.html')

@app.route('/admin/students')
def manage_students():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    students = db.get_all_students()
    student_rows = ''
    for student in students:
        student_rows += f'<tr><td>{student["id"]}</td><td>{student["name"]}</td><td>{student["email"]}</td><td>{student["phone"]}</td><td>{student["status"]}</td><td><a href="/admin/update_student?id={student["id"]}">Edit</a> | <a href="/admin/delete_student?id={student["id"]}">Delete</a></td></tr>'
    return render_template('manage_students.html', student_rows=student_rows)

@app.route('/admin/add_student', methods=['POST'])
def add_student():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    name = request.form.get('name')
    email = request.form.get('email')
    phone = request.form.get('phone')
    address = request.form.get('address')
    admission_date = request.form.get('admission_date')
    username = email
    password = 'student123'
    user_id = db.create_user(username, password, 'student')
    db.create_student(user_id, name, email, phone, address, admission_date)
    return redirect(url_for('manage_students'))

@app.route('/admin/update_student', methods=['GET', 'POST'])
def update_student():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    if request.method == 'POST':
        student_id = request.form.get('id')
        name = request.form.get('name')
        email = request.form.get('email')
        phone = request.form.get('phone')
        address = request.form.get('address')
        room_id = request.form.get('room_id') or None
        status = request.form.get('status', 'active')
        db.update_student(student_id, name, email, phone, address, room_id, status)
        return redirect(url_for('manage_students'))
    else:
        student_id = request.args.get('id')
        student = db.get_student_by_id(student_id)
        if not student:
            return "Student not found", 404
        return render_template('update_student.html', student=student)

@app.route('/admin/delete_student', methods=['GET', 'POST'])
def delete_student():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    if request.method == 'POST':
        student_id = request.form.get('id')
        db.delete_student(student_id)
        return redirect(url_for('manage_students'))
    else:
        student_id = request.args.get('id')
        student = db.get_student_by_id(student_id)
        if not student:
            return "Student not found", 404
        return render_template('delete_student.html', student=student)

@app.route('/admin/rooms')
def manage_rooms():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    rooms = db.get_all_rooms()
    students = db.get_all_students()
    room_rows = ''
    for room in rooms:
        allocated = 'Yes' if room['allocated_to'] else 'No'
        actions = f'<button class="btn btn-primary btn-sm" onclick="allocateRoom({room["id"]})">Allocate</button> <button class="btn btn-danger btn-sm" onclick="vacateRoom({room["id"]})">Vacate</button>' if not room['allocated_to'] else f'<button class="btn btn-danger btn-sm" onclick="vacateRoom({room["id"]})">Vacate</button>'
        room_rows += f'<tr><td>{room["room_number"]}</td><td>{room["capacity"]}</td><td>{room["type"]}</td><td>{room["status"]}</td><td>{allocated}</td><td>{actions}</td></tr>'
    student_options = ''
    for student in students:
        if student['status'] == 'active' and not student['room_id']:
            student_options += f'<option value="{student["id"]}">{student["name"]} ({student["email"]})</option>'
    return render_template('manage_rooms.html', room_rows=room_rows, student_options=student_options)

@app.route('/admin/add_room', methods=['POST'])
def add_room():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    room_number = request.form.get('room_number')
    capacity = int(request.form.get('capacity', 1))
    room_type = request.form.get('type')
    db.create_room(room_number, capacity, room_type)
    return redirect(url_for('manage_rooms'))

@app.route('/admin/allocate_room', methods=['POST'])
def allocate_room():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    room_id = request.form.get('room_id')
    student_id = request.form.get('student_id')
    db.allocate_room(room_id, student_id)
    return redirect(url_for('manage_rooms'))

@app.route('/admin/vacate_room', methods=['POST'])
def vacate_room():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    room_id = request.form.get('room_id')
    db.vacate_room(room_id)
    return redirect(url_for('manage_rooms'))

@app.route('/admin/fees')
def manage_fees():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    fees = db.get_all_fees()
    students = db.get_all_students()
    fee_rows = ''
    for fee in fees:
        student_name = next((s['name'] for s in students if s['id'] == fee['student_id']), 'Unknown')
        fee_rows += f'<tr><td>{student_name} ({fee["student_id"]})</td><td>{fee["amount"]}</td><td>{fee["type"]}</td><td>{fee["due_date"]}</td><td>{fee["status"]}</td><td><a href="/admin/update_fee?id={fee["id"]}">Update</a></td></tr>'
    student_options = ''
    for student in students:
        student_options += f'<option value="{student["id"]}">{student["name"]} ({student["email"]})</option>'
    return render_template('manage_fees.html', fee_rows=fee_rows, student_options=student_options)

@app.route('/admin/add_fee', methods=['POST'])
def add_fee():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    student_id = request.form.get('student_id')
    amount = float(request.form.get('amount', 0))
    fee_type = request.form.get('type', 'monthly')
    due_date = request.form.get('due_date')
    db.create_fee(student_id, amount, fee_type, due_date)
    return redirect(url_for('manage_fees'))

@app.route('/admin/update_fee', methods=['GET', 'POST'])
def update_fee():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    if request.method == 'POST':
        fee_id = request.form.get('id')
        status = request.form.get('status', 'paid')
        payment_date = datetime.now().date() if status == 'paid' else None
        db.update_fee_status(fee_id, status, payment_date)
        return redirect(url_for('manage_fees'))
    else:
        fee_id = request.args.get('id')
        fee = db.get_fee_by_id(fee_id)
        if not fee:
            return "Fee not found", 404
        return render_template('update_fee.html', fee=fee)

@app.route('/admin/visitors')
def manage_visitors():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    visitors = db.get_all_visitors()
    students = db.get_all_students()
    visitor_rows = ''
    for visitor in visitors:
        student_name = next((s['name'] for s in students if s['id'] == visitor['student_id']), 'Unknown')
        visitor_rows += f'<tr><td>{visitor["name"]}</td><td>{student_name} ({visitor["student_id"]})</td><td>{visitor["address"]}</td><td>{visitor["relation"]}</td><td>{visitor["time_in"]}</td><td>{visitor["time_out"] or "N/A"}</td><td><a href="/admin/update_visitor?id={visitor["id"]}">Update</a></td></tr>'
    student_options = ''
    for student in students:
        student_options += f'<option value="{student["id"]}">{student["name"]} ({student["email"]})</option>'
    return render_template('manage_visitors.html', visitor_rows=visitor_rows, student_options=student_options)

@app.route('/admin/add_visitor', methods=['POST'])
def add_visitor():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    student_id = request.form.get('student_id')
    name = request.form.get('name')
    address = request.form.get('address')
    relation = request.form.get('relation')
    time_in = datetime.now()
    db.create_visitor(student_id, name, address, relation, time_in)
    return redirect(url_for('manage_visitors'))

@app.route('/admin/update_visitor', methods=['GET', 'POST'])
def update_visitor():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    if request.method == 'POST':
        visitor_id = request.form.get('id')
        time_out = datetime.now()
        db.update_visitor_time_out(visitor_id, time_out)
        return redirect(url_for('manage_visitors'))
    else:
        visitor_id = request.args.get('id')
        visitor = db.get_visitor_by_id(visitor_id)
        if not visitor:
            return "Visitor not found", 404
        return render_template('update_visitor.html', visitor=visitor)

@app.route('/admin/complaints')
def manage_complaints():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    complaints = db.get_all_complaints()
    complaint_rows = ''
    for complaint in complaints:
        complaint_rows += f'<tr><td>{complaint["subject"]}</td><td>{complaint["description"]}</td><td>{complaint["status"]}</td><td>{complaint["submitted_at"]}</td><td><a href="/admin/update_complaint?id={complaint["id"]}">Update</a></td></tr>'
    return render_template('manage_complaints.html', complaint_rows=complaint_rows)

@app.route('/admin/update_complaint', methods=['GET', 'POST'])
def update_complaint():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    if request.method == 'POST':
        complaint_id = request.form.get('id')
        status = request.form.get('status', 'resolved')
        resolved_at = datetime.now() if status == 'resolved' else None
        db.update_complaint_status(complaint_id, status, resolved_at)
        return redirect(url_for('manage_complaints'))
    else:
        complaint_id = request.args.get('id')
        complaint = db.get_complaint_by_id(complaint_id)
        if not complaint:
            return "Complaint not found", 404
        return render_template('update_complaint.html', complaint=complaint)

@app.route('/admin/reports')
def reports():
    if session.get('role') != 'admin':
        return redirect(url_for('login'))
    students = db.get_all_students()
    rooms = db.get_all_rooms()
    fees = db.get_fees_by_student(None)
    visitors = db.get_visitors_by_student(None)
    student_report = '<h3>Student List</h3><table class="table"><thead><tr><th>ID</th><th>Name</th><th>Email</th><th>Status</th></tr></thead><tbody>'
    for s in students:
        student_report += f'<tr><td>{s["id"]}</td><td>{s["name"]}</td><td>{s["email"]}</td><td>{s["status"]}</td></tr>'
    student_report += '</tbody></table>'
    room_report = '<h3>Room Allocation Report</h3><table class="table"><thead><tr><th>Room Number</th><th>Type</th><th>Allocated To</th></tr></thead><tbody>'
    for r in rooms:
        allocated_to = r['allocated_to'] or 'N/A'
        room_report += f'<tr><td>{r["room_number"]}</td><td>{r["type"]}</td><td>{allocated_to}</td></tr>'
    room_report += '</tbody></table>'
    payment_report = '<h3>Payment Report</h3><table class="table"><thead><tr><th>Student ID</th><th>Amount</th><th>Type</th><th>Status</th></tr></thead><tbody>'
    for f in fees:
        payment_report += f'<tr><td>{f["student_id"]}</td><td>{f["amount"]}</td><td>{f["type"]}</td><td>{f["status"]}</td></tr>'
    payment_report += '</tbody></table>'
    visitor_report = '<h3>Visitor Log Report</h3><table class="table"><thead><tr><th>Name</th><th>Student ID</th><th>Time In</th><th>Time Out</th></tr></thead><tbody>'
    for v in visitors:
        time_out = v['time_out'] or 'N/A'
        visitor_report += f'<tr><td>{v["name"]}</td><td>{v["student_id"]}</td><td>{v["time_in"]}</td><td>{time_out}</td></tr>'
    visitor_report += '</tbody></table>'
    return render_template('reports.html', student_report=student_report, room_report=room_report, payment_report=payment_report, visitor_report=visitor_report)

@app.route('/student/complaints')
def student_complaints():
    if session.get('role') != 'student':
        return redirect(url_for('login'))
    user_id = session.get('user_id')
    student = db.get_student_by_id(user_id)
    complaints = db.get_all_complaints()
    complaint_rows = ''
    for c in complaints:
        if c['student_id'] == student['id']:
            complaint_rows += f'<tr><td>{c["subject"]}</td><td>{c["description"]}</td><td>{c["status"]}</td></tr>'
    return render_template('student_complaints.html', complaint_rows=complaint_rows)

@app.route('/student/add_complaint', methods=['POST'])
def add_complaint():
    if session.get('role') != 'student':
        return redirect(url_for('login'))
    user_id = session.get('user_id')
    student = db.get_student_by_id(user_id)
    subject = request.form.get('subject')
    description = request.form.get('description')
    db.create_complaint(student['id'], subject, description)
    return redirect(url_for('student_complaints'))

@app.route('/student/visitors')
def student_visitors():
    if session.get('role') != 'student':
        return redirect(url_for('login'))
    user_id = session.get('user_id')
    student = db.get_student_by_id(user_id)
    visitors = db.get_visitors_by_student(student['id'])
    visitor_rows = ''
    for v in visitors:
        visitor_rows += f'<tr><td>{v["name"]}</td><td>{v["address"]}</td><td>{v["relation"]}</td><td>{v["time_in"]}</td><td>{v["time_out"] or "N/A"}</td></tr>'
    return render_template('student_visitors.html', visitor_rows=visitor_rows)

@app.route('/student/add_visitor', methods=['POST'])
def add_visitor_student():
    if session.get('role') != 'student':
        return redirect(url_for('login'))
    user_id = session.get('user_id')
    student = db.get_student_by_id(user_id)
    name = request.form.get('name')
    address = request.form.get('address')
    relation = request.form.get('relation')
    time_in = datetime.now()
    db.create_visitor(student['id'], name, address, relation, time_in)
    return redirect(url_for('student_visitors'))

if __name__ == '__main__':
    app.run(debug=True, host='0.0.0.0', port=int(os.environ.get("PORT", 8000)))
